//
// (c) 2020 wesolutions GmbH
// All rights reserved.
//

import QtQuick 2.12

import wesual.Ui 1.0

import wesual.Ui.Private 1.0

UiLineEditBase {
    id : uiLineEdit

    property real fontSizeFactor : 1.0

    property alias readOnly  : edit.readOnly
    property alias inputMask : edit.inputMask
    property alias validator : edit.validator

    property alias padding : edit.padding
    property alias leftPadding : edit.leftPadding
    property alias rightPadding : edit.rightPadding
    property alias topPadding : edit.topPadding
    property alias bottomPadding : edit.bottomPadding

    property alias font : edit.font
    property color color : UiColors.getColor(UiColors.DarkGrey80)
    property alias echoMode : edit.echoMode

    property int minimumWidth : 80

    implicitWidth : Math.max(minimumWidth, edit.implicitWidth)
    implicitHeight : edit.implicitHeight

    textInput : edit
    baselineOffset : edit.baselineOffset

    Text {
        id : placeHolder

        padding : uiLineEdit.padding
        topPadding : uiLineEdit.topPadding
        bottomPadding : uiLineEdit.bottomPadding
        leftPadding : uiLineEdit.leftPadding
        rightPadding: uiLineEdit.rightPadding
        visible : uiLineEdit.empty && edit.text.length < 1
        color : uiLineEdit.enabled
                ? UiColors.getColor(UiColors.MediumGrey33)
                : UiColors.getColor(UiColors.DisabledLightGrey13)
        text : uiLineEdit.placeHolderText
        wrapMode : Text.NoWrap
        elide : Qt.ElideRight

        font : UiFonts.getFont(UiFonts.Light, Math.round(14 * fontSizeFactor))

        anchors.fill : parent
    }

    TextInput {
        id : edit

        readonly property bool overflow : implicitWidth > width

        clip : true
        topPadding : 4
        bottomPadding : 4
        color : {
            if (!uiLineEdit.enabled) {
                return UiColors.getColor(UiColors.MediumLightGrey27);
            } else if (uiLineEdit.readOnly) {
                return UiColors.getColor(UiColors.MediumGrey45);
            } else {
                return uiLineEdit.color;
            }
        }
        text : uiLineEdit.text

        activeFocusOnTab  : true
        selectByMouse     : true
        selectionColor    : UiColors.getColor(UiColors.SelectionGreen)
        selectedTextColor : UiColors.getColor(UiColors.White)

        font : UiFonts.getFont(UiFonts.Light, Math.round(14 * fontSizeFactor))

        anchors.fill : parent

        // Fade out overflow when not editing
        layer.enabled : !activeFocus && overflow
        layer.smooth : true
        layer.effect : ShaderEffect {
            property real low : 1.0 - 24 / Math.max(1, edit.width)

            fragmentShader: "
                uniform lowp sampler2D source;
                uniform lowp float qt_Opacity;
                uniform lowp float low;
                varying highp vec2 qt_TexCoord0;
                void main() {
                    lowp vec4 p = texture2D(source, qt_TexCoord0);
                    gl_FragColor = p *
                        (1.0 - smoothstep(low, 1.0, qt_TexCoord0.x)) *
                        qt_Opacity;
                }"
        }
    }

    Rectangle {
        id : background

        anchors.fill : parent
        anchors.leftMargin : -6
        anchors.rightMargin : -6

        color : UiColors.getColor(UiColors.White, 0)
        border {
            width : 0
            color : UiColors.getColor(UiColors.ControlGrey20)
        }
        z : -1

        states : [
            State {
                name : "focus"
                when : uiLineEdit.activeFocus && !uiLineEdit.readOnly

                PropertyChanges {
                    target       : background
                    border.width : 1
                    border.color : UiColors.getColor(UiColors.SelectionGreen)
                    color : UiColors.getColor(UiColors.White, 1)
                }
            },
            State {
                name : "hover"
                when : uiLineEdit.hovered && !uiLineEdit.readOnly

                PropertyChanges {
                    target       : background
                    border.width : 1
                    border.color : UiColors.getColor(UiColors.HoverGreen)
                    color : UiColors.getColor(UiColors.White, 1)
                }
            }
        ]
    }

    UiTextContextMenu {
        textEditor             : edit
        onContextMenuRequested : edit.forceActiveFocus()
    }
}
